<?php
namespace App\Core;

use PDO;
use PDOException;

final class DB {
  private static ?PDO $pdo = null;

  public static function init(array $cfg): void {
    if (self::$pdo) return;
    $dsn = sprintf('mysql:host=%s;port=%s;dbname=%s;charset=utf8mb4',
      $cfg['host'], $cfg['port'], $cfg['database']
    );
    try {
      self::$pdo = new PDO($dsn, $cfg['username'], $cfg['password'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
      ]);
    } catch (PDOException $e) {
      // In production: log
      throw $e;
    }
  }

  public static function pdo(): PDO {
    if (!self::$pdo) throw new \RuntimeException('DB not initialized.');
    return self::$pdo;
  }

  public static function query(string $sql, array $params = []): array {
    $stmt = self::pdo()->prepare($sql);
    $stmt->execute($params);
    return $stmt->fetchAll();
  }

  public static function exec(string $sql, array $params = []): int {
    $stmt = self::pdo()->prepare($sql);
    $stmt->execute($params);
    return $stmt->rowCount();
  }

  public static function lastInsertId(): string {
    return self::pdo()->lastInsertId();
  }
}
