<?php
namespace App\Core;

final class Router {
  private array $routes = [];

  public function get(string $path, callable|array $handler): void {
    $this->add('GET', $path, $handler);
  }
  public function post(string $path, callable|array $handler): void {
    $this->add('POST', $path, $handler);
  }

  private function add(string $method, string $path, callable|array $handler): void {
    $this->routes[] = [$method, $this->normalize($path), $handler];
  }

  private function normalize(string $path): string {
    $path = '/' . trim($path, '/');
    return $path === '//' ? '/' : $path;
  }

  private function stripQuery(string $uri): string {
    return parse_url($uri, PHP_URL_PATH) ?? '/';
  }

  public function dispatch(string $method, string $uri): void {
    $path = $this->normalize($this->stripQuery($uri));

    foreach ($this->routes as [$m, $p, $h]) {
      if ($m !== $method) continue;

      // param route: /blog/{slug}
      $regex = preg_replace('#\{([a-zA-Z_][a-zA-Z0-9_]*)\}#', '(?P<$1>[^/]+)', $p);
      $regex = '#^' . $regex . '$#';

      if (preg_match($regex, $path, $matches)) {
        $params = array_filter($matches, 'is_string', ARRAY_FILTER_USE_KEY);
        return $this->invoke($h, $params);
      }
    }

    http_response_code(404);
    echo View::render('errors/404', ['path' => $path]);
  }

  private function invoke(callable|array $handler, array $params): void {
    if (is_callable($handler)) {
      $handler($params);
      return;
    }
    // [ControllerClass, 'method']
    [$class, $method] = $handler;
    $controller = new $class();
    $controller->$method($params);
  }
}
