<?php
declare(strict_types=1);

session_start();
define('BASE_PATH', dirname(__DIR__));

function e(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

$lock = BASE_PATH . '/storage/install.lock';
if (file_exists($lock)) { http_response_code(403); echo "Installer locked."; exit; }

$step = (int)($_GET['step'] ?? 1);
$error = null;

$requirements = [
  'PHP >= 8.1' => version_compare(PHP_VERSION, '8.1.0', '>='),
  'PDO' => extension_loaded('pdo'),
  'pdo_mysql' => extension_loaded('pdo_mysql'),
  'mbstring' => extension_loaded('mbstring'),
  'gd' => extension_loaded('gd'),
  'openssl' => extension_loaded('openssl'),
];

if ($step === 1) {
  // requirements page
} elseif ($step === 2 && $_SERVER['REQUEST_METHOD'] === 'POST') {
  $_SESSION['db'] = [
    'host' => trim($_POST['host'] ?? 'localhost'),
    'port' => trim($_POST['port'] ?? '3306'),
    'database' => trim($_POST['database'] ?? ''),
    'username' => trim($_POST['username'] ?? ''),
    'password' => (string)($_POST['password'] ?? ''),
  ];
  header('Location: ?step=3'); exit;
} elseif ($step === 3 && $_SERVER['REQUEST_METHOD'] === 'POST') {
  $_SESSION['admin'] = [
    'name' => trim($_POST['name'] ?? 'Admin'),
    'email' => trim($_POST['email'] ?? ''),
    'password' => (string)($_POST['password'] ?? ''),
  ];
  // run install
  $db = $_SESSION['db'] ?? null;
  $admin = $_SESSION['admin'] ?? null;
  if (!$db || !$admin) { $error = 'بيانات ناقصة'; $step = 1; }
  else {
    try {
      $dsn = sprintf('mysql:host=%s;port=%s;dbname=%s;charset=utf8mb4', $db['host'],$db['port'],$db['database']);
      $pdo = new PDO($dsn, $db['username'], $db['password'], [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      ]);

      // migrations
      $sql = file_get_contents(BASE_PATH . '/database/schema.sql');
      $pdo->exec($sql);

      // seed categories
      $pdo->exec("INSERT IGNORE INTO work_categories (slug,title,sort_order) VALUES
        ('web','تصميم مواقع',1),
        ('graphic','جرافيك',2),
        ('apps','تطبيقات',3)
      ");

      // seed service placeholders
      $pdo->exec("INSERT IGNORE INTO services (id,title,excerpt,sort_order) VALUES
        (1,'تصميم مواقع','خدمة تصميم مواقع احترافية.',1),
        (2,'تسويق إلكتروني','إعلانات ممولة + سوشيال.',2),
        (3,'SEO','تحسين محركات البحث.',3)
      ");

      // create admin user
      if (empty($admin['email']) || empty($admin['password'])) throw new Exception('بيانات الأدمن غير مكتملة');
      $hash = password_hash($admin['password'], PASSWORD_BCRYPT);
      $stmt = $pdo->prepare("INSERT INTO users (name,email,password_hash,role,created_at) VALUES (?,?,?,?,NOW())");
      $stmt->execute([$admin['name'], $admin['email'], $hash, 'admin']);

      // create a demo client user
      $cHash = password_hash('123456', PASSWORD_BCRYPT);
      $pdo->prepare("INSERT INTO client_users (full_name,email,password_hash,created_at) VALUES (?,?,?,NOW())")
          ->execute(['Client Demo','client@demo.com',$cHash]);

      // write config/database.php
      $cfg = "<?php\nreturn " . var_export($db, true) . ";\n";
      if (!is_dir(BASE_PATH . '/config')) mkdir(BASE_PATH . '/config', 0775, true);
      file_put_contents(BASE_PATH . '/config/database.php', $cfg);

      // ensure storage
      if (!is_dir(BASE_PATH . '/storage/uploads/tickets')) mkdir(BASE_PATH . '/storage/uploads/tickets', 0775, true);

      // lock
      if (!is_dir(BASE_PATH . '/storage')) mkdir(BASE_PATH . '/storage', 0775, true);
      file_put_contents($lock, 'locked:' . date('c'));

      header('Location: success.php'); exit;

    } catch (Throwable $t) {
      $error = $t->getMessage();
      $step = 1;
    }
  }
}

?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8"/>
  <meta name="viewport" content="width=device-width,initial-scale=1"/>
  <title>Installer</title>
  <style>
    body{font-family:system-ui;margin:0;background:#fafafa}
    .wrap{max-width:780px;margin:30px auto;padding:16px}
    .card{background:#fff;border:1px solid #eee;border-radius:16px;padding:16px}
    .btn{display:inline-block;padding:10px 14px;border-radius:10px;border:1px solid #ddd;background:#fff;cursor:pointer;text-decoration:none;color:#111}
    input{width:100%;padding:10px;border:1px solid #ddd;border-radius:10px}
    label{display:block;margin:10px 0 6px}
    .muted{color:#666;font-size:14px}
    .bad{color:#b00020}
    .ok{color:#0a7a0a}
  </style>
</head>
<body>
<div class="wrap">
  <div class="card">
    <h2 style="margin-top:0">تنصيب الموقع</h2>
    <?php if ($error): ?><div class="bad"><?= e($error) ?></div><?php endif; ?>

    <?php if ($step === 1): ?>
      <h3>1) متطلبات السيرفر</h3>
      <ul>
      <?php foreach ($requirements as $k=>$v): ?>
        <li class="<?= $v?'ok':'bad' ?>"><?= e($k) ?>: <?= $v?'OK':'Missing' ?></li>
      <?php endforeach; ?>
      </ul>
      <a class="btn" href="?step=2">التالي</a>

    <?php elseif ($step === 2): ?>
      <h3>2) إعداد قاعدة البيانات</h3>
      <form method="post" action="?step=2">
        <label>Host</label><input name="host" value="localhost"/>
        <label>Port</label><input name="port" value="3306"/>
        <label>Database</label><input name="database" />
        <label>Username</label><input name="username" />
        <label>Password</label><input name="password" type="password" />
        <button class="btn" type="submit" style="margin-top:12px">حفظ</button>
      </form>

    <?php elseif ($step === 3): ?>
      <h3>3) إنشاء حساب الأدمن</h3>
      <p class="muted">بعد إنشاء الأدمن سيتم قفل التنصيب تلقائيًا.</p>
      <form method="post" action="?step=3">
        <label>الاسم</label><input name="name" value="Admin"/>
        <label>البريد</label><input name="email" />
        <label>كلمة المرور</label><input name="password" type="password" />
        <button class="btn" type="submit" style="margin-top:12px">تنصيب الآن</button>
      </form>
    <?php endif; ?>
  </div>

  <div class="card" style="margin-top:12px">
    <div class="muted">بعد التنصيب: ادخل /admin أو /client (حساب تجريبي: client@demo.com / 123456)</div>
  </div>
</div>
</body>
</html>
